<?php
/**
 * Debug Cache Content for Keywords Autolinker
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Include necessary files
require_once(ABSPATH . 'wp-config.php');
require_once(ABSPATH . 'wp-includes/wp-db.php');

// Check cache content
global $wpdb;
$cache_table = $wpdb->prefix . 'affiliate_hub_autolinker_cache';

echo "<h2>Keywords Autolinker Cache Debug</h2>";

// Check if table exists
$table_exists = $wpdb->get_var("SHOW TABLES LIKE '$cache_table'") === $cache_table;

if (!$table_exists) {
    echo "<p style='color: red;'>❌ Cache table does not exist: $cache_table</p>";
    
    // Try to create table
    echo "<p>Attempting to create table...</p>";
    
    $sql = "CREATE TABLE $cache_table (
        id int(11) NOT NULL AUTO_INCREMENT,
        post_id bigint(20) unsigned NOT NULL,
        keyword varchar(255) NOT NULL,
        affiliate_link text NOT NULL,
        priority int(11) DEFAULT 100,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY post_id (post_id),
        KEY keyword (keyword),
        KEY priority (priority)
    ) DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;";
    
    $result = $wpdb->query($sql);
    if ($result !== false) {
        echo "<p style='color: green;'>✅ Table created successfully</p>";
    } else {
        echo "<p style='color: red;'>❌ Failed to create table: " . $wpdb->last_error . "</p>";
    }
} else {
    echo "<p style='color: green;'>✅ Cache table exists: $cache_table</p>";
}

// Get cache content
$cache_data = $wpdb->get_results("SELECT * FROM $cache_table ORDER BY keyword ASC");

echo "<h3>Cache Content</h3>";
echo "<p>Total entries: " . count($cache_data) . "</p>";

if (!empty($cache_data)) {
    echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
    echo "<tr><th>ID</th><th>Post ID</th><th>Keyword</th><th>Link</th><th>Priority</th><th>Created</th></tr>";
    
    foreach ($cache_data as $row) {
        echo "<tr>";
        echo "<td>{$row->id}</td>";
        echo "<td>{$row->post_id}</td>";
        echo "<td><strong>{$row->keyword}</strong></td>";
        echo "<td>" . substr($row->affiliate_link, 0, 50) . "...</td>";
        echo "<td>{$row->priority}</td>";
        echo "<td>{$row->created_at}</td>";
        echo "</tr>";
    }
    
    echo "</table>";
} else {
    echo "<p style='color: orange;'>⚠️ Cache is empty. No keywords found.</p>";
    
    // Check if there are affiliate links
    $affiliate_links = $wpdb->get_results("SELECT ID, post_title FROM {$wpdb->posts} WHERE post_type = 'affiliate_link' AND post_status = 'publish' LIMIT 10");
    
    echo "<h4>Available Affiliate Links</h4>";
    if (!empty($affiliate_links)) {
        echo "<ul>";
        foreach ($affiliate_links as $link) {
            echo "<li>ID: {$link->ID} - {$link->post_title}</li>";
            
            // Check if link has keywords
            $keywords = get_post_meta($link->ID, '_autolink_keywords', true);
            if (!empty($keywords)) {
                echo "<ul><li>Keywords: " . implode(', ', $keywords) . "</li></ul>";
            } else {
                echo "<ul><li>No keywords set</li></ul>";
            }
        }
        echo "</ul>";
    } else {
        echo "<p>No affiliate links found</p>";
    }
}

// Test manual cache rebuild
echo "<h3>Manual Cache Rebuild Test</h3>";
echo "<p><a href='?rebuild_cache=1' style='background: #0073aa; color: white; padding: 10px; text-decoration: none;'>Rebuild Cache Now</a></p>";

if (isset($_GET['rebuild_cache'])) {
    echo "<div style='background: #fff3cd; padding: 10px; border: 1px solid #ffeaa7;'>";
    echo "<h4>Rebuilding Cache...</h4>";
    
    try {
        // Include the cache class
        require_once(dirname(__FILE__) . '/../includes/Modules/KeywordsAutolinker/Cache.php');
        
        $cache = new \AffiliateHub\Modules\KeywordsAutolinker\Cache();
        $cache->rebuild_all_cache();
        
        echo "<p style='color: green;'>✅ Cache rebuilt successfully!</p>";
        echo "<p><a href='?'>Refresh page to see results</a></p>";
        
    } catch (Exception $e) {
        echo "<p style='color: red;'>❌ Error rebuilding cache: " . $e->getMessage() . "</p>";
    }
    
    echo "</div>";
}

echo "<h3>Debug Information</h3>";
echo "<ul>";
echo "<li>WordPress Version: " . get_bloginfo('version') . "</li>";
echo "<li>PHP Version: " . PHP_VERSION . "</li>";
echo "<li>Database Prefix: " . $wpdb->prefix . "</li>";
echo "<li>Current User: " . wp_get_current_user()->user_login . "</li>";
echo "</ul>";

?>
