<?php
/**
 * Test Script for Real-time Keyword Conflict Detection
 * 
 * This script tests the AJAX endpoint for keyword conflict detection
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    require_once(dirname(__DIR__) . '/../../wp-config.php');
}

// Test data
$test_cases = [
    [
        'keyword' => 'test keyword',
        'post_id' => 123,
        'expected' => 'no conflict (new keyword)'
    ],
    [
        'keyword' => 'existing keyword',
        'post_id' => 456,
        'expected' => 'potential conflict if exists in cache'
    ],
    [
        'keyword' => '',
        'post_id' => 789,
        'expected' => 'error (empty keyword)'
    ]
];

echo "<h2>Keyword Conflict Detection Test</h2>\n";
echo "<p>Testing AJAX endpoint: affiliate_hub_check_keyword_conflict</p>\n";

// Test the Cache class directly
require_once(dirname(__DIR__) . '/includes/Modules/KeywordsAutolinker/Cache.php');

try {
    $cache = new \AffiliateHub\Modules\KeywordsAutolinker\Cache();
    $cached_keywords = $cache->get_cached_keywords();
    
    echo "<h3>Current Cache Status:</h3>\n";
    echo "<p>Total cached keywords: " . count($cached_keywords) . "</p>\n";
    
    if (!empty($cached_keywords)) {
        echo "<h4>Sample cached keywords:</h4>\n";
        echo "<ul>\n";
        foreach (array_slice($cached_keywords, 0, 5) as $item) {
            echo "<li>'{$item['keyword']}' (Post ID: {$item['post_id']})</li>\n";
        }
        echo "</ul>\n";
    }
    
    echo "<h3>Conflict Detection Logic Test:</h3>\n";
    
    // Test case-insensitive matching
    if (!empty($cached_keywords)) {
        $test_keyword = $cached_keywords[0]['keyword'];
        echo "<p><strong>Testing with existing keyword:</strong> '{$test_keyword}'</p>\n";
        
        $variations = [
            strtolower($test_keyword),
            strtoupper($test_keyword),
            ucfirst(strtolower($test_keyword))
        ];
        
        foreach ($variations as $variation) {
            $conflict_found = false;
            foreach ($cached_keywords as $cached) {
                if (strcasecmp($cached['keyword'], $variation) === 0) {
                    $conflict_found = true;
                    break;
                }
            }
            echo "<p>'{$variation}' → " . ($conflict_found ? "CONFLICT DETECTED" : "No conflict") . "</p>\n";
        }
    }
    
} catch (Exception $e) {
    echo "<p style='color: red;'>Error testing cache: " . $e->getMessage() . "</p>\n";
}

echo "<h3>AJAX Endpoint Information:</h3>\n";
echo "<p><strong>Action:</strong> affiliate_hub_check_keyword_conflict</p>\n";
echo "<p><strong>Required Parameters:</strong></p>\n";
echo "<ul>\n";
echo "<li>keyword (string): The keyword to check</li>\n";
echo "<li>post_id (int): Current post ID to exclude from conflict check</li>\n";
echo "<li>nonce (string): WordPress security nonce</li>\n";
echo "</ul>\n";

echo "<p><strong>Response Format:</strong></p>\n";
echo "<pre>\n";
echo "Success (no conflict):\n";
echo "{\n";
echo "  \"success\": true,\n";
echo "  \"data\": {\n";
echo "    \"conflict\": false,\n";
echo "    \"message\": \"Keyword is available\"\n";
echo "  }\n";
echo "}\n\n";

echo "Success (conflict found):\n";
echo "{\n";
echo "  \"success\": true,\n";
echo "  \"data\": {\n";
echo "    \"conflict\": true,\n";
echo "    \"existing_link_id\": 123,\n";
echo "    \"existing_link_name\": \"Example Link\",\n";
echo "    \"existing_link_url\": \"http://example.com/link\",\n";
echo "    \"message\": \"Keyword 'test' already exists in link: Example Link (ID: 123)\"\n";
echo "  }\n";
echo "}\n";
echo "</pre>\n";

echo "<h3>Usage in JavaScript:</h3>\n";
echo "<pre>\n";
echo "$.ajax({\n";
echo "    url: affiliateHubAutolinker.ajaxUrl,\n";
echo "    type: 'POST',\n";
echo "    data: {\n";
echo "        action: 'affiliate_hub_check_keyword_conflict',\n";
echo "        keyword: 'test keyword',\n";
echo "        post_id: 123,\n";
echo "        nonce: affiliateHubAutolinker.nonce\n";
echo "    },\n";
echo "    success: function(response) {\n";
echo "        if (response.success && response.data.conflict) {\n";
echo "            // Show conflict warning\n";
echo "            console.log(response.data.message);\n";
echo "        }\n";
echo "    }\n";
echo "});\n";
echo "</pre>\n";

echo "<p><em>Note: This system provides real-time feedback as users type keywords, helping prevent conflicts before saving.</em></p>\n";
?>
